<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\EvaluasiModel;
use App\Models\GuruModel;
use App\Models\Periode;
use Dompdf\Dompdf;
use Dompdf\Options;

class EvaluasiController extends BaseController
{
    protected $evaluasiModel;
    protected $guruModel;
    protected $periode;

    public function __construct()
        {
        $this->evaluasiModel = new EvaluasiModel();
        $this->guruModel = new GuruModel();
        $this->periode = new Periode();
        }

    private function loadEvaluasiView($viewName)
        {
            $builder = $this->evaluasiModel->builder();
            $data['evaluasi'] = $builder
                ->select('evaluasi_kinerja.*, guru.nama_guru AS nama_guru, evaluasi_kinerja.alasan_penolakan, periode.nama_periode AS periode, periode.tahun_ajaran AS tahun_ajaran')
                ->join('guru', 'guru.id = evaluasi_kinerja.id_guru', 'left')
                ->join('periode', 'periode.id = evaluasi_kinerja.periode', 'left')      
                ->orderBy('evaluasi_kinerja.created_at', 'DESC')
                ->get()
                ->getResultArray();
            $data['daftarGuru'] = $this->guruModel->findAll();

            return view("evaluasi/$viewName", $data);
        }

        public function cetakPdf($id)
{
    $db = \Config\Database::connect();

    $evaluasi = $db->table('evaluasi_kinerja')
        ->where('id', $id)
        ->get()->getRow();

    if (!$evaluasi) {
        return redirect()->back()->with('error', 'Data evaluasi tidak ditemukan.');
    }

    $guru = $db->table('guru')
        ->where('id', $evaluasi->id_guru)
        ->get()->getRow();

    $indikator = $db->table('nilai_indikator')
        ->where('id_evaluasi', $id)
        ->get()->getResultArray();

    $nilaiMap = [];
    foreach ($indikator as $n) {
        $nilaiMap[$n['indikator']] = $n['nilai'];
    }

    // Predikat
    $total = $evaluasi->nilai;
    if ($total < 70) $predikat = 'Kurang';
    elseif ($total <= 80) $predikat = 'Cukup';
    elseif ($total <= 95) $predikat = 'Baik';
    else $predikat = 'Baik Sekali';

    // Siapkan view HTML
    $html = view('evaluasi/pdf_detail', [
        'e' => $evaluasi,
        'guru' => $guru,
        'nilaiMap' => $nilaiMap,
        'predikat' => $predikat
    ]);

    // Load Dompdf
    $options = new Options();
    $options->set('defaultFont', 'Arial');
    $dompdf = new Dompdf($options);
    $dompdf->loadHtml($html);
    $dompdf->setPaper('A4', 'portrait');
    $dompdf->render();

    // Output PDF
    return $this->response
        ->setHeader('Content-Type', 'application/pdf')
        ->setBody($dompdf->output());
}

    public function index()
        {
            $query = $this->evaluasiModel
                ->select('evaluasi_kinerja.*, guru.nama_guru AS nama_guru, evaluasi_kinerja.alasan_penolakan')
                ->join('guru', 'guru.id = evaluasi_kinerja.id_guru', 'left');
            $data['evaluasi'] = $query->orderBy('evaluasi_kinerja.created_at', 'DESC')->findAll();
            $data['daftarGuru'] = $this->guruModel->findAll();

            $role = session()->get('role');
            switch ($role) {
                case 'admin':
                    return view('evaluasi/evaluasi_admin', $data);
                case 'kepsek':
                    return view('evaluasi/evaluasi_kepsek', $data);
                case 'guru':
                    return view('evaluasi/evaluasi_guru', $data);
                default:
                    return redirect()->to('/login')->with('error', 'Role tidak dikenali.');
            }
        }

    public function evaluasi_admin()
        {
            return $this->loadEvaluasiView('evaluasi_admin');
        }

    public function evaluasi_kepsek()
        {
            $model = new \App\Models\EvaluasiModel();

            $data['evaluasi'] = $model
                ->select('evaluasi_kinerja.*, guru.nama_guru, guru.id AS id_guru, periode.nama_periode AS periode, periode.tahun_ajaran')
                ->join('guru', 'guru.id = evaluasi_kinerja.id_guru', 'left')
                ->join('periode', 'periode.id = evaluasi_kinerja.periode', 'left')
                ->orderBy('evaluasi_kinerja.created_at', 'DESC')
                ->findAll();

            // Ambil RPP terakhir jika kosong
            foreach ($data['evaluasi'] as &$e) {
                if (empty($e['rpp'])) {
                    $lastRpp = $model
                        ->select('rpp')
                        ->where('id_guru', $e['id_guru'])
                        ->where('rpp IS NOT NULL')
                        ->where('rpp != ""')
                        ->orderBy('id', 'DESC')
                        ->first();

                    $e['rpp_terakhir'] = $lastRpp['rpp'] ?? null;
                } else {
                    $e['rpp_terakhir'] = $e['rpp'];
                }
            }

            return view('evaluasi/evaluasi_kepsek', $data);
        }

    public function approve($id)
        {
            $model = new EvaluasiModel();

            // Update status menjadi 'disetujui'
            $model->update($id, ['status' => 'disetujui']);

            return redirect()->to('evaluasi/evaluasi_kinerja')->with('success', 'Evaluasi berhasil disetujui dan siap untuk dinilai.');
        }

    public function reject($id)
        {
            $evaluasiModel = new \App\Models\EvaluasiModel();
            $dataEvaluasi = $evaluasiModel->find($id);

            if (!$dataEvaluasi) {
                return redirect()->back()->with('error', 'Data tidak ditemukan.');
            }

            if ($this->request->getMethod() === 'post') {
                $alasan = $this->request->getPost('alasan_penolakan');

                if (empty($alasan)) {
                    return redirect()->back()->with('error', 'Alasan penolakan harus diisi.');
                }

                $evaluasiModel->update($id, [
                    'status' => 'Ditolak',
                    'alasan_penolakan' => $alasan,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);

                return redirect()->to('/evaluasi/evaluasi_kepsek')->with('success', 'Laporan berhasil ditolak.');
            }

            return view('evaluasi/tolak_evaluasi', ['evaluasi' => $dataEvaluasi]);
        }

    public function evaluasi_kinerja()
{
    $evaluasiData = $this->evaluasiModel
        ->select('evaluasi_kinerja.*, 
                evaluasi_kinerja.rpp AS rpp_file,
                evaluasi_kinerja.foto_kegiatan,
                evaluasi_kinerja.absensi_bulanan,
                guru.nama_guru AS nama_guru, 
                periode.nama_periode AS periode, 
                periode.tahun_ajaran AS tahun_ajaran,
                guru.id AS id_guru')
        ->join('guru', 'guru.id = evaluasi_kinerja.id_guru', 'left')
        ->join('periode', 'periode.id = evaluasi_kinerja.periode', 'left')
        ->where('evaluasi_kinerja.status', 'disetujui')
        ->orderBy('evaluasi_kinerja.created_at', 'DESC')
        ->findAll();

    // Ambil RPP terakhir Januari/Juli untuk setiap guru
    $evaluasiModel = new \App\Models\EvaluasiModel();
    foreach ($evaluasiData as &$e) {
        $rppTerakhir = $evaluasiModel
            ->select('rpp')
            ->where('id_guru', $e['id_guru'])
            ->whereIn('periode', function ($builder) {
                return $builder->select('id')
                    ->from('periode')
                    ->whereIn('LOWER(nama_periode)', ['januari', 'juli']);
            })
            ->where('rpp IS NOT NULL')
            ->orderBy('id', 'DESC')
            ->first();

        $e['rpp_terakhir'] = $rppTerakhir['rpp'] ?? null;
    }

    $data['evaluasi'] = $evaluasiData;

    return view('evaluasi/evaluasi_kinerja', $data);
}


    public function evaluasi_guru()
{
    $session = session();
    $login = $session->get('login');

    if (!$login || $login['role'] !== 'guru') {
        return redirect()->to('/login')->with('error', 'Akses ditolak.');
    }

    $idUser = $login['id'];
    $evaluasiModel = $this->evaluasiModel;

    $evaluasi = $evaluasiModel
        ->select('evaluasi_kinerja.*, guru.nama_guru AS nama_guru, periode.nama_periode AS periode, periode.tahun_ajaran AS tahun_ajaran')
        ->join('users', 'users.id_guru = evaluasi_kinerja.id_guru', 'left')
        ->join('guru', 'guru.id = evaluasi_kinerja.id_guru', 'left')
        ->join('periode', 'periode.id = evaluasi_kinerja.periode', 'left')
        ->where('users.id', $idUser)
        ->orderBy('evaluasi_kinerja.created_at', 'DESC')
        ->findAll();

    // Tambahkan rpp_terakhir ke setiap entri
    foreach ($evaluasi as &$e) {
        $lastRpp = $evaluasiModel
            ->select('rpp')
            ->where('id_guru', $e['id_guru'])
            ->where('rpp IS NOT NULL')
            ->where('rpp != ""')
            ->orderBy('id', 'DESC')
            ->first();

        $e['rpp_terakhir'] = $lastRpp['rpp'] ?? null;
    }

    return view('evaluasi/evaluasi_guru', ['evaluasi' => $evaluasi]);
}

    public function create()
        {
            $session = session();
            $loginData = $session->get('login');

            if (!$loginData || !isset($loginData['id'])) {
                return redirect()->to('/login')->with('error', 'Silakan login terlebih dahulu');
            }

            $periode = new \App\Models\Periode();
            $evaluasiModel = new \App\Models\EvaluasiModel();

            $idGuru = $loginData['id_guru'];

            // Cari RPP terakhir dari periode Januari/Juli
            $lastRpp = $evaluasiModel
                ->select('rpp')
                ->where('id_guru', $idGuru)
                ->whereIn('periode', function ($builder) {
                    return $builder->select('id')
                        ->from('periode')
                        ->whereIn('LOWER(nama_periode)', ['januari', 'juli']);
                })
                ->where('rpp IS NOT NULL')
                ->orderBy('id', 'DESC')
                ->first();

            $data = [
                'idGuruLogin' => $idGuru,
                'namaGuruLogin' => $loginData['username'],
                'periode' => $periode->findAll(),
                'lastRpp' => $lastRpp['rpp'] ?? null
            ];

            return view('evaluasi/create', $data);
        }

    public function store()
        {
            $validation = \Config\Services::validation();

            $rules = [
                'id_guru'            => 'required',
                'periode'            => 'required',
               'rpp'                 => 'if_exist|mime_in[rpp,application/pdf,application/msword,application/vnd.openxmlformats-officedocument.wordprocessingml.document]',
                'foto_kegiatan'      => 'uploaded[foto_kegiatan]|mime_in[foto_kegiatan,image/jpg,image/jpeg,image/png]',
                'dokumen_absensi'    => 'uploaded[dokumen_absensi]|mime_in[dokumen_absensi,application/pdf,application/msword,application/vnd.openxmlformats-officedocument.wordprocessingml.document,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet,image/jpg,image/jpeg,image/png]',
                'dokumen_kedisiplinan' => 'uploaded[dokumen_kedisiplinan]|mime_in[dokumen_kedisiplinan,image/jpg,image/jpeg,image/png]',
                'tema_bulanan'       => 'uploaded[tema_bulanan]|mime_in[tema_bulanan,image/jpg,image/jpeg,image/png]'
            ];

            if (!$this->validate($rules)) {
                return redirect()->back()->withInput()->with('errors', $validation->getErrors());
            }

            // Cek apakah sudah ada evaluasi dengan id_guru dan periode yang sama
            $idGuru = $this->request->getPost('id_guru');
            $periodeId = $this->request->getPost('periode');
            $sudahAda = $this->evaluasiModel
                ->where('id_guru', $idGuru)
                ->where('periode', $periodeId)
                ->countAllResults();

            if ($sudahAda > 0) {
                return redirect()->back()->withInput()->with('error', 'Evaluasi untuk periode ini sudah pernah diajukan.');
            }

            // Ambil file
           $rppFile = $this->request->getFile('rpp');
            $rppName = null;
            if ($rppFile && $rppFile->isValid() && !$rppFile->hasMoved()) {
                $rppName = $rppFile->getRandomName();
                $rppFile->move('uploads/rpp', $rppName);
            }
            $fotoFile             = $this->request->getFile('foto_kegiatan');
            $absensiFile          = $this->request->getFile('dokumen_absensi');
            $kedisiplinanFile     = $this->request->getFile('dokumen_kedisiplinan');
            $temaBulananFile      = $this->request->getFile('tema_bulanan');

            // Random file name
            $fotoName             = $fotoFile->getRandomName();
            $absensiName          = $absensiFile->getRandomName();
            $kedisiplinanName     = $kedisiplinanFile->getRandomName();
            $temaBulananName      = $temaBulananFile->getRandomName();

            // Simpan file ke direktori
            $fotoFile->move('uploads/foto', $fotoName);
            $absensiFile->move('uploads/absensi', $absensiName);
            $kedisiplinanFile->move('uploads/kedisiplinan', $kedisiplinanName);
            $temaBulananFile->move('uploads/tema_bulanan', $temaBulananName);

            // Simpan ke database
            $this->evaluasiModel->insert([
                'id_guru'             => $idGuru,
                'periode'             => $periodeId,
                'rpp'                 => $rppName,
                'foto_kegiatan'       => $fotoName,
                'dokumen_absensi'     => $absensiName,
                'dokumen_kedisiplinan'=> $kedisiplinanName,
                'tema_bulanan'        => $temaBulananName,
                'status'              => 'Diajukan',
                'created_at'          => (new \DateTime('now', new \DateTimeZone('Asia/Jakarta')))->format('Y-m-d H:i:s'),
            ]);

            return redirect()->to('/evaluasi/evaluasi_guru')->with('success', 'Data evaluasi berhasil ditambahkan!');
        }

    public function edit($id)
        {
            $evaluasiModel = new EvaluasiModel();
            $evaluasi = $evaluasiModel->find($id);

            if (!$evaluasi) {
                throw new \CodeIgniter\Exceptions\PageNotFoundException('Data tidak ditemukan');
            }

            // Tambahkan data periode
            $periodeModel = new \App\Models\Periode();
            $data = [
                'evaluasi' => $evaluasi,
                'periode' => $periodeModel->findAll()
            ];

            return view('evaluasi/edit', $data);
        }

    public function update($id)
{
    $evaluasiModel = new \App\Models\EvaluasiModel();
    $evaluasi = $evaluasiModel->find($id);

    if (!$evaluasi) {
        return redirect()->back()->with('error', 'Data evaluasi tidak ditemukan.');
    }

    // Ambil data file
    $fileRpp      = $this->request->getFile('rpp');
    $fileFoto     = $this->request->getFile('foto_kegiatan');
    $fileAbsensi  = $this->request->getFile('dokumen_absensi');
    $fileDisiplin = $this->request->getFile('dokumen_kedisiplinan');
    $fileTema     = $this->request->getFile('tema_bulanan');

    $data = [
        'periode'       => $this->request->getPost('periode'),
        'nama_kegiatan' => $this->request->getPost('nama_kegiatan'),
        'status'        => 'Diajukan',
        'updated_at'    => date('Y-m-d H:i:s'),
    ];

    // === RPP ===
    if ($fileRpp && $fileRpp->isValid() && !$fileRpp->hasMoved()) {
        $newRppName = $fileRpp->getRandomName();
        $fileRpp->move('uploads/rpp', $newRppName);
        if (!empty($evaluasi['rpp']) && file_exists('uploads/rpp/' . $evaluasi['rpp'])) {
            unlink('uploads/rpp/' . $evaluasi['rpp']);
        }
        $data['rpp'] = $newRppName;
    }

    // === Foto Kegiatan ===
    if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
        $newFotoName = $fileFoto->getRandomName();
        $fileFoto->move('uploads/foto_kegiatan', $newFotoName);
        if (!empty($evaluasi['foto_kegiatan']) && file_exists('uploads/foto_kegiatan/' . $evaluasi['foto_kegiatan'])) {
            unlink('uploads/foto_kegiatan/' . $evaluasi['foto_kegiatan']);
        }
        $data['foto_kegiatan'] = $newFotoName;
    }

    // === Absensi ===
    if ($fileAbsensi && $fileAbsensi->isValid() && !$fileAbsensi->hasMoved()) {
        $newAbsensiName = $fileAbsensi->getRandomName();
        $fileAbsensi->move('uploads/absensi', $newAbsensiName);
        if (!empty($evaluasi['dokumen_absensi']) && file_exists('uploads/absensi/' . $evaluasi['dokumen_absensi'])) {
            unlink('uploads/absensi/' . $evaluasi['dokumen_absensi']);
        }
        $data['dokumen_absensi'] = $newAbsensiName;
    }

    // === Kedisiplinan ===
    if ($fileDisiplin && $fileDisiplin->isValid() && !$fileDisiplin->hasMoved()) {
        $newDisiplinName = $fileDisiplin->getRandomName();
        $fileDisiplin->move('uploads/kedisiplinan', $newDisiplinName);
        if (!empty($evaluasi['dokumen_kedisiplinan']) && file_exists('uploads/kedisiplinan/' . $evaluasi['dokumen_kedisiplinan'])) {
            unlink('uploads/kedisiplinan/' . $evaluasi['dokumen_kedisiplinan']);
        }
        $data['dokumen_kedisiplinan'] = $newDisiplinName;
    }

    // === Tema Bulanan ===
    if ($fileTema && $fileTema->isValid() && !$fileTema->hasMoved()) {
        $newTemaName = $fileTema->getRandomName();
        $fileTema->move('uploads/tema_bulanan', $newTemaName);
        if (!empty($evaluasi['tema_bulanan']) && file_exists('uploads/tema_bulanan/' . $evaluasi['tema_bulanan'])) {
            unlink('uploads/tema_bulanan/' . $evaluasi['tema_bulanan']);
        }
        $data['tema_bulanan'] = $newTemaName;
    }

    // Simpan perubahan
    $evaluasiModel->update($id, $data);

    return redirect()->to('/evaluasi/evaluasi_guru')->with('success', 'Evaluasi berhasil diperbarui dan diajukan ulang.');
}

        // Hapus data evaluasi berdasarkan ID
    public function delete($id)
        {
            $data = $this->evaluasiModel->find($id);

            if (!$data) {
                return redirect()->back()->with('error', 'Data tidak ditemukan.');
            }

            $this->evaluasiModel->delete($id);

            $role = session()->get('login')['role'];

            switch ($role) {
                case 'admin':
                    return redirect()->to('/evaluasi/evaluasi_admin')->with('success', 'Data berhasil dihapus.');
                case 'kepsek':
                    return redirect()->to('/evaluasi/evaluasi_kepsek')->with('success', 'Data berhasil dihapus.');
                case 'guru':
                    return redirect()->to('/evaluasi/evaluasi_guru')->with('success', 'Data berhasil dihapus.');
                default:
                    return redirect()->to('/login')->with('error', 'Role tidak dikenali.');
            }
        }
    
   public function formNilai($id)
{
    $evaluasiModel = new \App\Models\EvaluasiModel();
    $guruModel = new \App\Models\GuruModel();

    // Ambil data evaluasi berdasarkan ID
    $evaluasi = $evaluasiModel->find($id);

    if (!$evaluasi) {
        return redirect()->back()->with('error', 'Data evaluasi tidak ditemukan.');
    }

    // Ambil data guru berdasarkan id_guru dari tabel evaluasi_kinerja
    $guru = $guruModel->find($evaluasi['id_guru']);

    return view('evaluasi/form_nilai', [
        'evaluasi' => $evaluasi,
        'guru'     => $guru ?? [] // aman jika kosong
    ]);
}

 public function simpanNilai()
{
    $model = new \App\Models\EvaluasiModel();
    $db = \Config\Database::connect();
    $post = $this->request->getPost();

    $id = $post['id'] ?? null;
    if (!$id) {
        return redirect()->back()->with('error', 'ID evaluasi tidak ditemukan.');
    }

    $indikator = $post['indikator'] ?? [];
    $tidakHadir = (int)($post['tidak_hadir'] ?? 0);
    $tepatWaktu = (int)($post['tepat_waktu'] ?? 0);
    $totalHari = 20;

    // Skor hadir
    $hadir = $totalHari - $tidakHadir;
    $persenHadir = ($hadir / $totalHari) * 100;
    $skorHadir = $persenHadir >= 95 ? 5 : ($persenHadir >= 85 ? 4 : ($persenHadir >= 75 ? 3 : ($persenHadir >= 60 ? 2 : 1)));

    // Skor tepat waktu
    $persenTepat = ($tepatWaktu / $totalHari) * 100;
    $skorTepat = $persenTepat >= 95 ? 5 : ($persenTepat >= 85 ? 4 : ($persenTepat >= 75 ? 3 : ($persenTepat >= 60 ? 2 : 1)));

    // Hitung skor rata-rata per kompetensi
    $skorPedagogik = (
        (int)($indikator['pedagogik1'] ?? 0) +
        (int)($indikator['pedagogik2'] ?? 0) +
        (int)($indikator['pedagogik3'] ?? 0) +
        (int)($indikator['pedagogik4'] ?? 0)
    ) / 4;

    $skorProfesional = (
        (int)($indikator['profesional1'] ?? 0) +
        (int)($indikator['profesional2'] ?? 0) +
        (int)($indikator['profesional3'] ?? 0)
    ) / 3;

    $skorKepribadian = (
        (int)($indikator['kepribadian1'] ?? 0) +
        (int)($indikator['kepribadian2'] ?? 0) +
        $skorHadir + $skorTepat
    ) / 4;

    $skorSosial = (
        (int)($indikator['sosial1'] ?? 0) +
        (int)($indikator['sosial2'] ?? 0) +
        (int)($indikator['sosial3'] ?? 0)
    ) / 3;

    // Konversi ke bobot nilai
    $pedagogik   = round(($skorPedagogik / 5) * 20, 2);
    $profesional = round(($skorProfesional / 5) * 20, 2);
    $kepribadian = round(($skorKepribadian / 5) * 40, 2);
    $sosial      = round(($skorSosial / 5) * 20, 2);

    $totalNilai = round($pedagogik + $profesional + $kepribadian + $sosial, 2);

    // Simpan ke tabel evaluasi_kinerja
    $model->update($id, [
        'pedagogik'           => $pedagogik,
        'profesional'         => $profesional,
        'kepribadian'         => $kepribadian,
        'sosial'              => $sosial,
        'nilai'               => $totalNilai,
        'jumlah_tidak_hadir'  => $tidakHadir,
        'jumlah_tepat_waktu'  => $tepatWaktu,
        'absensi_bulanan'     => $skorHadir,
        'catatan_kepsek'      => $post['catatan'] ?? null,
        'updated_at'          => date('Y-m-d H:i:s'),
    ]);

    // Reset nilai indikator lama
    $db->table('nilai_indikator')->where('id_evaluasi', $id)->delete();

    // Siapkan data nilai indikator dari form
    $dataNilai = [];
    foreach ($indikator as $key => $nilaiInd) {
        $dataNilai[] = [
            'id_evaluasi' => $id,
            'indikator'   => $key,
            'nilai'       => (int)$nilaiInd,
            'created_at'  => date('Y-m-d H:i:s'),
        ];
    }

    // Tambahkan nilai hadir dan tepat waktu sebagai indikator tambahan
    $dataNilai[] = [
        'id_evaluasi' => $id,
        'indikator'   => 'kepribadian_hadir',
        'nilai'       => $skorHadir,
        'created_at'  => date('Y-m-d H:i:s'),
    ];
    $dataNilai[] = [
        'id_evaluasi' => $id,
        'indikator'   => 'kepribadian_tepat',
        'nilai'       => $skorTepat,
        'created_at'  => date('Y-m-d H:i:s'),
    ];

    // Simpan batch nilai indikator
    $db->table('nilai_indikator')->insertBatch($dataNilai);

    return redirect()->to(base_url('evaluasi/evaluasi_kinerja'))->with('success', 'Penilaian berhasil disimpan.');
}


}
